/*
 * Copyright 2012 Jesse Long.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package allinone;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

class MultiURLClassLoader
        extends ClassLoader
{
    private final List<URL> urls = new LinkedList<URL>();

    public MultiURLClassLoader(ClassLoader parent, Collection<URL> urls)
    {
        super(parent);
        this.urls.addAll(urls);
    }

    private static String getManifestAttributeValue(Manifest manifest, String attributeName)
    {
        Attributes attributes = manifest.getMainAttributes();

        if (attributes == null){
            return null;
        }

        return attributes.getValue(attributeName);
    }

    @Override
    protected Class<?> findClass(String name)
            throws ClassNotFoundException
    {
        String classResourceName = name.replace(".", "/").concat(".class");

        URL u = findResource(classResourceName);

        if (u == null){
            throw new ClassNotFoundException();
        }

        byte[] classBytes;

        try {
            InputStream in = u.openStream();
            try {
                ByteArrayOutputStream baos = new ByteArrayOutputStream(10000);

                byte[] buffer = new byte[10240];
                int rdbytes;

                while ((rdbytes = in.read(buffer)) >= 0){
                    if (rdbytes > 0){
                        baos.write(buffer, 0, rdbytes);
                    }
                }

                classBytes = baos.toByteArray();
            }finally{
                in.close();
            }
        }catch (IOException e){
            throw new ClassNotFoundException("Error reading class bytes", e);
        }

        String packageName;

        {
            int idx = name.lastIndexOf(".");

            if (idx < 0){
                packageName = "";
            }else{
                packageName = name.substring(0, idx);
            }
        }

        if (packageName.trim().length() != 0){
            if (getPackage(packageName) == null){
                URL manifestUrl = findResource("META-INF/MANIFEST.MF");

                if (manifestUrl != null){

                    Manifest manifest;

                    try {
                        InputStream in = manifestUrl.openStream();
                        try {
                            manifest = new Manifest();
                            manifest.read(in);
                        }finally{
                            in.close();
                        }
                    }catch (IOException e){
                        throw new ClassNotFoundException("Error opening manifest stream", e);
                    }

                    String sealedString = getManifestAttributeValue(manifest, "Sealed");
                    URL sealedUrl;

                    if ("true".equals(sealedString)){
                        String baseUrlString = u.toString();

                        if (baseUrlString.endsWith(classResourceName)){
                            baseUrlString = baseUrlString.substring(0, baseUrlString.length() - classResourceName.length());
                        }

                        try {
                            sealedUrl = new URL(u, baseUrlString);
                        }catch (MalformedURLException e){
                            throw new ClassNotFoundException("Error building sealed URL", e);
                        }
                    }else{
                        sealedUrl = null;
                    }

                    definePackage(packageName,
                            getManifestAttributeValue(manifest, "Specification-Title"),
                            getManifestAttributeValue(manifest, "Specification-Version"),
                            getManifestAttributeValue(manifest, "Specification-Vendor"),
                            getManifestAttributeValue(manifest, "Implementation-Title"),
                            getManifestAttributeValue(manifest, "Implementation-Version"),
                            getManifestAttributeValue(manifest, "Implementation-Vendor"),
                            sealedUrl);
                }

            }
        }

        Class<?> clazz = defineClass(name, classBytes, 0, classBytes.length, null);

        return clazz;
    }

    protected URL findResource(URL baseUrl, String name)
    {
        String prefix = baseUrl.toString();

        if (!prefix.endsWith("/")){
            prefix = prefix.concat("/");
        }

        if (name.startsWith("/")){
            name = name.substring(1);
        }

        try {
            URL u = new URL(baseUrl, prefix + name);

            try {
                URLConnection conn = u.openConnection();

                conn.getInputStream().close();

                return u;
            }catch (IOException e){
                return null;
            }
        }catch (MalformedURLException e){
            return null;
        }
    }

    @Override
    protected URL findResource(String name)
    {
        synchronized (urls){

            for (URL baseUrl : urls){

                URL u = findResource(baseUrl, name);

                if (u != null){
                    urls.remove(baseUrl);
                    urls.add(0, baseUrl);
                    return u;
                }

            }
            return null;
        }
    }

    @Override
    protected Enumeration<URL> findResources(String name)
            throws IOException
    {
        synchronized (urls){

            List<URL> retv = new LinkedList<URL>();

            for (URL baseUrl : urls){

                URL u = findResource(baseUrl, name);

                if (u != null){
                    retv.add(u);
                }
            }
            
            return Collections.enumeration(retv);
        }
    }
}
